﻿using System;
using System.Collections.Generic;

namespace SlXnaAppSample
{
    /// <summary>
    /// IServiceProvider をアプリケーションに実装します。この型は、App.Services プロパティを通じて公開され、
    /// ContentManagers など、IServiceProvider にアクセスする必要がある型で使用できます。
    /// </summary>
    public class AppServiceProvider : IServiceProvider
    {
        // サービス自体に行われるサービスの型のマップ
        private readonly Dictionary<Type, object> services = new Dictionary<Type, object>();

        /// <summary>
        /// 新しいサービスをサービス プロバイダーに追加します。
        /// </summary>
        /// <param name="serviceType">追加するサービスの型。</param>
        /// <param name="service">サービス オブジェクト自体。</param>
        public void AddService(Type serviceType, object service)
        {
            // 入力を検証します
            if (serviceType == null)
                throw new ArgumentNullException("serviceType");
            if (service == null)
                throw new ArgumentNullException("service");
            if (!serviceType.IsAssignableFrom(service.GetType()))
                throw new ArgumentException("service does not match the specified serviceType");

            // サービスをディクショナリに追加します
            services.Add(serviceType, service);
        }

        /// <summary>
        /// サービス プロバイダーからサービスを取得します。
        /// </summary>
        /// <param name="serviceType">取得するサービスの型。</param>
        /// <returns>指定の型に登録されたサービス オブジェクト。</returns>
        public object GetService(Type serviceType)
        {
            // 入力を検証します
            if (serviceType == null)
                throw new ArgumentNullException("serviceType");

            // ディクショナリからサービスを取得します
            return services[serviceType];
        }

        /// <summary>
        /// サービス プロバイダーからサービスを削除します。
        /// </summary>
        /// <param name="serviceType">削除するサービスの型。</param>
        public void RemoveService(Type serviceType)
        {
            // 入力を検証します
            if (serviceType == null)
                throw new ArgumentNullException("serviceType");

            // ディクショナリからサービスを削除します
            services.Remove(serviceType);
        }
    }
}
