﻿using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Navigation;
using Microsoft.Phone.Controls;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

namespace SlXnaAppSample
{
    public partial class GamePage : PhoneApplicationPage
    {
        /// <summary>Contentプロジェクトのコンテンツを管理するオブジェクト。</summary>
        ContentManager contentManager;

        /// <summary>Silverlightの上でXNAのゲームループを発生させるタイマー。</summary>
        GameTimer timer;

        /// <summary>SilverlightのUIコントロールを2D画像に変換するオブジェクト。</summary>
        UIElementRenderer elementRenderer;

        /// <summary>2D画像を描画するポリゴン。</summary>
        SpriteBatch spriteBatch;

        /// <summary>3Dモデルを格納するオブジェクト。</summary>
        Model model;

        /// <summary>3Dモデルの回転角度。</summary>
        float yaw, pitch;

        /// <summary>
        /// 新しいインスタンスを生成します。
        /// </summary>
        public GamePage()
        {
            InitializeComponent();

            // アプリケーションからコンテンツマネージャーを取得します
            contentManager = (Application.Current as App).Content;

            // このページのタイマーを作成します
            timer = new GameTimer();
            timer.UpdateInterval = TimeSpan.FromTicks(333333);
            timer.Update += OnUpdate;
            timer.Draw += OnDraw;

            // イベントハンドラを登録します
            LayoutUpdated += new EventHandler(GamePage_LayoutUpdated);
        }

        /// <summary>
        /// 画面のレイアウトが更新されたときに呼び出されます。
        /// </summary>
        private void GamePage_LayoutUpdated(object sender, EventArgs e)
        {
            // 更新されたレイアウトを基にしてUIElementRendererクラスを再構築します
            if (elementRenderer != null)
            {
                elementRenderer.Dispose();
            }
            elementRenderer = new UIElementRenderer(this,
                SharedGraphicsDeviceManager.Current.GraphicsDevice.Viewport.Width,
                SharedGraphicsDeviceManager.Current.GraphicsDevice.Viewport.Height);
        }

        /// <summary>
        /// この画面に遷移したときに呼び出されます。
        /// </summary>
        protected override void OnNavigatedTo(NavigationEventArgs e)
        {
            // グラフィックスデバイスの共有モードを設定してXNAレンダリングを有効にします
            SharedGraphicsDeviceManager.Current.GraphicsDevice.SetSharingMode(true);

            // UIコントロールを描画する2Dポリゴンを作成します
            spriteBatch = new SpriteBatch(SharedGraphicsDeviceManager.Current.GraphicsDevice);

            // 3Dモデルを読み込みます
            model = contentManager.Load<Model>("csharp-tan");

            // タイマーを開始します
            timer.Start();

            base.OnNavigatedTo(e);
        }

        /// <summary>
        /// 他の画面に遷移したときに呼び出されます。
        /// </summary>
        protected override void OnNavigatedFrom(NavigationEventArgs e)
        {
            // タイマーを停止します
            timer.Stop();

            // グラフィックスデバイスの共有モードを設定してXNAレンダリングを無効にします
            SharedGraphicsDeviceManager.Current.GraphicsDevice.SetSharingMode(false);

            base.OnNavigatedFrom(e);
        }

        /// <summary>
        /// XNAのゲームロジックが更新されるたびに呼び出されます。
        /// </summary>
        private void OnUpdate(object sender, GameTimerEventArgs e)
        {
            // 経過時間からXNAの3Dモデルの回転角度を算出します
            yaw += (float)e.ElapsedTime.TotalMilliseconds * MathHelper.ToRadians(0.05f);
        }

        /// <summary>
        /// XNAのゲームロジックが描画されるたびに呼び出されます。
        /// </summary>
        private void OnDraw(object sender, GameTimerEventArgs e)
        {
            // 画面をクリアします
            SharedGraphicsDeviceManager.Current.GraphicsDevice.Clear(Color.Black);

            // SilverlightのUIコントロールをXNAで描画します
            elementRenderer.Render();
            spriteBatch.Begin();
            spriteBatch.Draw(elementRenderer.Texture, Vector2.Zero, Color.White);
            spriteBatch.End();

            // SpriteBatchクラスで変更されたレンダーステートを元に戻します
            SharedGraphicsDeviceManager.Current.GraphicsDevice.BlendState = BlendState.Opaque;
            SharedGraphicsDeviceManager.Current.GraphicsDevice.DepthStencilState = DepthStencilState.Default;

            // 3Dモデルの位置と向きと傾きを定義して、回転角度を合成します
            Matrix world = Matrix.CreateWorld(Vector3.Zero, Vector3.Forward, Vector3.Up) *
                Matrix.CreateFromYawPitchRoll(yaw, pitch, 0);

            // カメラの位置と向きと傾きを定義します
            Matrix view = Matrix.CreateLookAt(new Vector3(-6, 16, 60), new Vector3(-6, 16, 0), Vector3.Up);

            // カメラのレンズの視野角と縦横比と撮影範囲を定義します
            Matrix projection = Matrix.CreatePerspectiveFieldOfView(MathHelper.ToRadians(45),
                SharedGraphicsDeviceManager.Current.GraphicsDevice.Viewport.AspectRatio, 0.1f, 1000f);

            // 3Dモデルを描画します
            model.Draw(world, view, projection);
        }

        /// <summary>
        /// Silverlightのリストコントロールが選択されたときに呼び出されます。
        /// </summary>
        private void ListBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            // 選択項目と同じ名称の3Dモデルの部位を発光させます
            string selectedName = partsListBox.SelectedValue as string;
            foreach (ModelMesh mesh in model.Meshes)
            {
                foreach (BasicEffect effect in mesh.Effects)
                {
                    effect.EmissiveColor = (selectedName == mesh.Name) ?
                        Color.Red.ToVector3() : Vector3.Zero;
                }
            }
        }

        /// <summary>
        /// Silverlightのボタンコントロールがクリックされたときに呼び出されます。
        /// </summary>
        private void Button_Click(object sender, RoutedEventArgs e)
        {
            // メッセージボックスを表示します
            MessageBox.Show("アッー！", "設定", MessageBoxButton.OK);
        }

        /// <summary>
        /// Silverlightのパネルコントロールがタッチされたときに呼び出されます。
        /// </summary>
        private void Grid_ManipulationDelta(object sender, ManipulationDeltaEventArgs e)
        {
            // ジェスチャーによる平面移動を3Dモデルの回転角度に変換します
            yaw = MathHelper.WrapAngle(yaw + (int)e.DeltaManipulation.Translation.X * 0.01f);
            pitch = MathHelper.WrapAngle(pitch + (int)e.DeltaManipulation.Translation.Y * 0.01f);
        }
    }
}